#!/bin/bash

chain_name=aembit-redirect-chain
aembit_group_id=$(id --group aembit_agent_proxy)
agent_proxy_port=38080
agent_dns_port=8053
container_cidr=$AEMBIT_DOCKER_CONTAINER_CIDR

update-alternatives --set iptables /usr/sbin/iptables-legacy
update-alternatives --set ip6tables /usr/sbin/ip6tables-legacy

# We encountered a bug where egress traffic from the Agent Proxy's DNS proxy will
# be routed back to itself. This rule prevents this bug.
# This was happening for the following reasons:
# 
# 1. The Client Workload and the Agent Proxy operate on shared networking infrastructure.
#    This means that they have the same IP address.
# 2. When the Client Workload initially connects, the DNS NAT rule in this file would
#    overwrite the destination IP and port to point at the Agent Proxy. 
# 3. After an iptables rule is evaluated, it is saved as a "connection" by conntrack.
#    It is not evaluated again, unless the connection in conntrack expires.
#    This means that the connection is saved in conntrack as:
#        - Original direction: <CW/AP IP>:<EPHEMERAL PORT X> -> <EXTERNAL DNS IP>:53
#        - Reply direction: 127.0.0.1:8053 -> <CW/AP IP>:<EPHEMERAL PORT X>
# 4. Since the Agent Proxy requests an ephemeral port on the host to handle the
#    outgoing DNS request, sometimes it is assigned a port that was previously
#    used and released by the Client Workload. However, since the NAT rule was already
#    evaluated, the mapping from step #3 was already saved, meaning that the outbound
#    DNS is sent to the Agent Proxy, forming a loop.
#
# To avoid this loop, in this line we set the NOTRACK target on outgoing DNS packets on
# the "raw" table, which is evaluated prior to conntrack. This causes conntrack to not
# try to associate the packet with any connection, even if one already exists.
# 
# See https://www.frozentux.net/iptables-tutorial/iptables-tutorial.html#STATEMACHINE
# for more information on conntrack.
iptables -t raw -A OUTPUT -p udp --dport 53 -m owner --gid-owner "${aembit_group_id}" -j NOTRACK

iptables  -t nat -N "${chain_name}"

# Ignore (return to the main chain) all packets from our agent proxy
iptables -t nat -A "${chain_name}" -m owner --gid-owner "${aembit_group_id}" -j RETURN

# Ignore TCP loopback
iptables -t nat -A "${chain_name}" -p tcp -o lo -j RETURN

# Redirect everything else to agent proxy port port
iptables -t nat -A "${chain_name}" -p  tcp -j REDIRECT --to-port "${agent_proxy_port}"

# Redirect DNS traffic to agent proxy DNS port, except for outbound traffic from systemd-resolved.
resolve_d_user_id=$(id --user systemd-resolve);
if [ -n "${resolve_d_user_id}" ]; then
    iptables -t nat -A "${chain_name}" -p udp --dport 53 -m owner --uid-owner "${resolve_d_user_id}" -j RETURN
fi
iptables -t nat -A "${chain_name}" -p udp --dport 53 -j REDIRECT --to "${agent_dns_port}"

# Add out chain 
iptables -t nat -A OUTPUT -j "${chain_name}"

# If we know the containers' CIDR block, preroute container traffic through agent proxy
if [ -n $container_cidr ]; then
    iptables -t nat -A PREROUTING -p tcp -s $container_cidr -j REDIRECT --to-port $agent_proxy_port
fi
